<?php

namespace App\Http\Controllers;

use App\Models\Producto;
use Illuminate\Http\Request;

class ProductoController extends Controller
{
    public function index(Request $request)
    {
        $query = Producto::query();

        if ($request->filled('categoria')) {
            $categoria = trim($request->input('categoria'));
            if ($categoria !== '' && strcasecmp($categoria, 'Todos') !== 0) {
                $query->where('categoria', 'like', '%' . $categoria . '%');
            }
        }

        if ($request->filled('valor')) {
            $valor = $request->input('valor');
            $query->where(function ($q) use ($valor) {
                $q->where('nombre', 'like', '%' . $valor . '%')
                  ->orWhere('codigo', 'like', '%' . $valor . '%');
            });
        }

        if ($request->has('per_page')) {
            return $query->paginate($request->input('per_page'));
        }

        return $query->get();
    }

    public function store(Request $request)
    {
        $request->validate([
            'codigo' => 'required|string|unique:productos,codigo',
            'nombre' => 'required|string',
            'categoria' => 'required|string',
            'precio_venta' => 'required|numeric|min:0',
            'costo' => 'required|numeric|min:0',
            'stock_inicial' => 'required|integer|min:0',
            'stock_minimo' => 'required|integer|min:0',
        ]);

        $producto = Producto::create($request->only([
            'codigo',
            'nombre',
            'categoria',
            'precio_venta',
            'costo',
            'stock_inicial',
            'stock_minimo',
        ]));

        return response()->json([
            'message' => 'Producto creado correctamente',
            'producto' => $producto,
        ], 201);
    }

    public function show(string $id)
    {
        $producto = Producto::find($id);

        if (!$producto) {
            return response()->json(['message' => 'Producto no encontrado'], 404);
        }

        return $producto;
    }

    public function update(Request $request, string $id)
    {
        $producto = Producto::find($id);

        if (!$producto) {
            return response()->json(['message' => 'Producto no encontrado'], 404);
        }

        $request->validate([
            'codigo' => 'sometimes|string|unique:productos,codigo,' . $id,
            'nombre' => 'sometimes|string',
            'categoria' => 'sometimes|string',
            'precio_venta' => 'sometimes|numeric|min:0',
            'costo' => 'sometimes|numeric|min:0',
            'stock_inicial' => 'sometimes|integer|min:0',
            'stock_minimo' => 'sometimes|integer|min:0',
        ]);

        $producto->update($request->only([
            'codigo',
            'nombre',
            'categoria',
            'precio_venta',
            'costo',
            'stock_inicial',
            'stock_minimo',
        ]));

        return response()->json([
            'message' => 'Producto actualizado correctamente',
            'producto' => $producto,
        ]);
    }

    public function destroy(string $id)
    {
        $producto = Producto::find($id);

        if (!$producto) {
            return response()->json(['message' => 'Producto no encontrado'], 404);
        }

        $producto->delete();

        return response()->json(['message' => 'Producto eliminado correctamente']);
    }
}
