<?php

namespace App\Http\Controllers;

use App\Models\Role;
use Illuminate\Http\Request;

class RoleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Role::with('permissions');

        if ($request->has('per_page')) {
            return $query->paginate($request->input('per_page'));
        }

        return $query->get();
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|unique:roles,nombre',
            'descripcion' => 'nullable|string',
            'permissions' => 'array'
        ]);

        $role = Role::create($request->only('nombre', 'descripcion'));

        if ($request->has('permissions')) {
            $role->permissions()->sync($request->permissions);
        }

        return response()->json(['message' => 'Rol creado correctamente', 'role' => $role->load('permissions')], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        return Role::with('permissions')->findOrFail($id);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $role = Role::findOrFail($id);

        $request->validate([
            'nombre' => 'required|unique:roles,nombre,' . $role->id,
            'descripcion' => 'nullable|string',
            'permissions' => 'array'
        ]);

        $role->update($request->only('nombre', 'descripcion'));

        if ($request->has('permissions')) {
            $role->permissions()->sync($request->permissions);
        }

        return response()->json(['message' => 'Rol actualizado correctamente', 'role' => $role->load('permissions')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $role = Role::findOrFail($id);
        $role->delete();

        return response()->json(['message' => 'Rol eliminado correctamente']);
    }
}
