<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = User::with(['role', 'sucursal']);

        if ($request->has('eliminado') && $request->eliminado === 'true') {
            $query->where('status', 'eliminado');
        } else {
            $query->where('status', '!=', 'eliminado');
        }

        if ($request->has('per_page')) {
            return $query->paginate($request->input('per_page'));
        }

        return $query->get();
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'usuario' => 'required|string|unique:users,usuario',
            'contrasena' => 'required|string|min:6',
            'role_id' => 'required|exists:roles,id',
            'sucursal_id' => 'nullable|exists:sucursals,id',
            'status' => 'required|in:activo,desactivado,eliminado',
        ]);

        $user = User::create([
            'usuario' => $request->usuario,
            'contrasena' => $request->contrasena,
            'role_id' => $request->role_id,
            'sucursal_id' => $request->sucursal_id,
            'status' => $request->status,
        ]);

        return response()->json([
            'message' => 'Usuario creado exitosamente',
            'user' => $user->load(['role', 'sucursal'])
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $user = User::find($id);
        if (!$user) {
            return response()->json(['message' => 'Usuario no encontrado'], 404);
        }
        return $user;
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $user = User::find($id);
        if (!$user) {
            return response()->json(['message' => 'Usuario no encontrado'], 404);
        }

        $request->validate([
            'usuario' => 'sometimes|string|unique:users,usuario,' . $id,
            'contrasena' => 'sometimes|string|min:6',
            'role_id' => 'sometimes|exists:roles,id',
            'sucursal_id' => 'nullable|exists:sucursals,id',
            'status' => 'sometimes|in:activo,desactivado,eliminado',
        ]);

        if ($request->has('usuario')) $user->usuario = $request->usuario;
        if ($request->has('role_id')) $user->role_id = $request->role_id;
        if ($request->has('sucursal_id')) $user->sucursal_id = $request->sucursal_id;
        if ($request->has('status')) $user->status = $request->status;
        if ($request->has('contrasena')) $user->contrasena = $request->contrasena;

        $user->save();

        return response()->json([
            'message' => 'Usuario actualizado',
            'user' => $user->load(['role', 'sucursal'])
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $user = User::find($id);
        if (!$user) {
            return response()->json(['message' => 'Usuario no encontrado'], 404);
        }

        $user->status = 'eliminado';
        $user->save();

        return response()->json(['message' => 'Usuario eliminado correctamente']);
    }
}
